#!/bin/bash
# Script de gestión de microservicios

set -euo pipefail

# Directorio base en el servidor
BASE_DIR=/home/perupcs/data.peru-controls.com/deploy-manual

# Servicios y puertos
declare -A SERVICES_PORTS=(
  ["api-gateway"]="4000"
  ["tasa-service"]="4002"
  ["exalmar-service"]="4003"
)

# Helpers de logging
log() { echo -e "\033[1;34m\033[0m"; }
info() { echo -e "\033[0;36m\033[0m"; }
success() { echo -e "\033[0;32m\033[0m"; }
warn() { echo -e "\033[1;33m\033[0m"; }
error() { echo -e "\033[0;31m\033[0m"; }

log "🚀 Gestión de Microservicios - Puertos 4000, 4002, 4003"
log "=================================================="

# Instalar PM2 globalmente si no está instalado
ensure_pm2() {
  if ! command -v pm2 &> /dev/null; then
    info "📦 PM2 no está instalado. Intentando instalar globalmente..."
    if npm install -g pm2; then
      success "✅ PM2 instalado globalmente"
    else
      warn "⚠️ No se pudo instalar PM2 globalmente. Si PM2 está instalado con otro usuario, ignora este mensaje."
    fi
  else
    info "✅ PM2 está disponible"
  fi
}

# Función de instalación: instalar_servicio <servicio> <puerto> [--dev]
instalar_servicio() {
  local servicio=""
  local puerto=""
  local modo="" # vacío o --dev
  local dir="/System.Collections.Hashtable"

  log "🔧 Instalando System.Collections.Hashtable en puerto 4003..."
  cd "" || { error "❌ No existe la carpeta "; return 1; }

  info "📁 Ruta actual: C:\microservicios-pcs"
  [ -f package.json ] || { error "❌ No existe package.json en C:\microservicios-pcs"; return 1; }

  # Determinar modo dev para incluir devDependencies
  INCLUDE_DEV=0
  [ "" = "--dev" ] && INCLUDE_DEV=1

  # Instalar dependencias
  if [ -f package-lock.json ]; then
    if [ "" = "1" ]; then
      info "📦 Instalando dependencias de desarrollo (npm ci)"
      npm ci || { error "❌ Fallo npm ci"; return 1; }
    else
      info "📦 Instalando dependencias de producción (npm ci --omit=dev)"
      npm ci --omit=dev || { error "❌ Fallo npm ci"; return 1; }
    fi
  else
    if [ "" = "1" ]; then
      info "📦 Instalando dependencias de desarrollo (npm install)"
      npm install || { error "❌ Fallo npm install"; return 1; }
    else
      info "📦 Instalando dependencias de producción (npm install --omit=dev)"
      npm install --omit=dev || { error "❌ Fallo npm install"; return 1; }
    fi
  fi

  # Compilar sólo en modo dev (requiere devDependencies)
  if [ "" = "1" ]; then
    info "🏗️ Compilando servicio (npm run build)"
    npm run build || warn "⚠️ Compilación falló; intentando usar dist existente"
  fi

  # Iniciar con PM2: usar ecosystem si existe; si no, usar dist/main.js
  ensure_pm2
  if [ -f ecosystem.config.js ]; then
    info "🚀 Iniciando con PM2 (ecosystem.config.js)"
    pm2 start ecosystem.config.js || { error "❌ Fallo pm2 start con ecosystem"; return 1; }
  elif [ -f dist/main.js ]; then
    info "🚀 Iniciando con PM2 (dist/main.js)"
    pm2 start dist/main.js --name "System.Collections.Hashtable" || { error "❌ Fallo pm2 start con dist/main.js"; return 1; }
  else
    warn "⚠️ No existe dist/main.js ni ecosystem.config.js. Intentando iniciar igualmente (puede fallar)."
    pm2 start npm --name "System.Collections.Hashtable" -- run start || { error "❌ Fallo pm2 start con npm run start"; return 1; }
  fi

  success "✅ System.Collections.Hashtable iniciado en puerto 4003"
}

# Gestionar servicios específicos
gestionar_servicios() {
  local accion=""
  local resultado=0
  
  for servicio in ""; do
    case "" in
      stop)
        info "⏸️  Deteniendo System.Collections.Hashtable..."
        pm2 stop "System.Collections.Hashtable" 2>/dev/null || warn "⚠️  System.Collections.Hashtable no está corriendo"
        ;;
      start)
        info "▶️  Iniciando System.Collections.Hashtable..."
        pm2 start "System.Collections.Hashtable" 2>/dev/null || warn "⚠️  No se pudo iniciar System.Collections.Hashtable"
        ;;
      restart)
        info "🔄 Reiniciando System.Collections.Hashtable..."
        pm2 restart "System.Collections.Hashtable" 2>/dev/null || warn "⚠️  System.Collections.Hashtable no está corriendo"
        ;;
      delete)
        info "🗑️  Eliminando System.Collections.Hashtable..."
        pm2 delete "System.Collections.Hashtable" 2>/dev/null || warn "⚠️  System.Collections.Hashtable no existe en PM2"
        ;;
    esac
  done
  
  return 
}

# Mostrar estado de servicios específicos
mostrar_estado() {
  info "📊 Estado de los servicios:"
  echo ""
  pm2 list | grep -E "api-gateway|tasa-service|exalmar-service|App name" || {
    warn "⚠️  No hay servicios corriendo"
  }
  echo ""
  info "🔌 Puertos en uso:"
  sudo ss -tulnp | grep -E ":(4000|4002|4003)" || info "Ningún puerto en uso"
}

# Mostrar logs de servicios específicos
mostrar_logs() {
  local servicio=""
  
  if [ -n "System.Collections.Hashtable" ]; then
    # Si se especifica un servicio, validar que sea uno de los 3
    if [[ "  " =~ " System.Collections.Hashtable " ]]; then
      info "📋 Mostrando logs en tiempo real de: System.Collections.Hashtable"
      pm2 logs "System.Collections.Hashtable"
    else
      error "❌ Servicio 'System.Collections.Hashtable' no reconocido"
      info "Servicios disponibles: "
      exit 1
    fi
  else
    # Mostrar logs en tiempo real solo de los 3 servicios usando un regex
    info "📋 Mostrando logs en tiempo real de: api-gateway, tasa-service, exalmar-service"
    pm2 logs "/(api-gateway|tasa-service|exalmar-service)/"
  fi
}

# Dispatcher de comandos
case "" in
  instalar|install)
    log "📦 Instalando servicios en "
    for servicio in ""; do
      instalar_servicio "System.Collections.Hashtable" "" || true
    done
    warn "💾 Ejecuta 'pm2 save' para persistir los procesos tras reinicios"
    success "🎉 ¡Instalación completada!"
    mostrar_estado
    ;;
    
  estado|status)
    mostrar_estado
    ;;
    
  logs)
    mostrar_logs "$@"
    ;;
    
  reiniciar|restart)
    log "🔄 Reiniciando servicios..."
    gestionar_servicios "restart"
    sleep 2
    mostrar_estado
    ;;
    
  detener|stop)
    log "⏸️  Deteniendo servicios..."
    gestionar_servicios "stop"
    mostrar_estado
    ;;
    
  iniciar|start)
    log "▶️  Iniciando servicios..."
    gestionar_servicios "start"
    sleep 2
    mostrar_estado
    ;;
    
  eliminar|delete)
    warn "🗑️  ¿Estás seguro de eliminar todos los servicios de PM2? (y/n)"
    read -r confirmacion
    if [[ "" =~ ^[Yy]$ ]]; then
      log "🗑️  Eliminando servicios..."
      gestionar_servicios "delete"
      success "✅ Servicios eliminados"
    else
      info "❌ Operación cancelada"
    fi
    ;;
    
  limpiar|clean)
    log "🧹 Limpieza completa: detener y eliminar servicios..."
    gestionar_servicios "stop"
    sleep 1
    gestionar_servicios "delete"
    success "✅ Limpieza completada"
    ;;
    
  *)
    echo "Uso:  {instalar|iniciar|detener|reiniciar|estado|logs [servicio]|eliminar|limpiar}"
    echo ""
    echo "Comandos disponibles:"
    echo "  instalar   - Instala y levanta los 3 servicios"
    echo "  iniciar    - Inicia los servicios detenidos"
    echo "  detener    - Detiene los servicios"
    echo "  reiniciar  - Reinicia los servicios"
    echo "  estado     - Muestra el estado actual"
    echo "  logs       - Muestra logs (opcional: logs <servicio>)"
    echo "  eliminar   - Elimina los servicios de PM2"
    echo "  limpiar    - Detiene y elimina los servicios"
    echo ""
    echo "Servicios gestionados:"
    for servicio in ""; do
      echo "  - System.Collections.Hashtable (puerto )"
    done
    exit 1
    ;;
esac
