"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var ExalmarService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExalmarService = void 0;
const common_1 = require("@nestjs/common");
const sequelize_1 = require("sequelize");
const exalmar_database_service_1 = require("./database/exalmar-database.service");
let ExalmarService = ExalmarService_1 = class ExalmarService {
    exalmarDatabaseService;
    logger = new common_1.Logger(ExalmarService_1.name);
    cache = new Map();
    CACHE_TTL = 180000;
    MAX_CONCURRENT_QUERIES = 5;
    embarcacionesConfig = {
        DONALFREDO: {
            nombre: 'DONALFREDO',
            codigoBD: '01864307SKY767C',
            tablas: [
                'reportesmelecto',
                'auxiliar1',
                'auxiliar2',
                'auxiliar3',
                'slave1',
            ],
        },
        MARIANGELLA: {
            nombre: 'MARIANGELLA',
            codigoBD: '01864306SKYF277',
            tablas: [
                'slave3',
                'auxiliar1',
                'auxiliar2',
                'auxiliar3',
                'slave1',
                'slave5',
            ],
        },
        CARMENCITA: {
            nombre: 'CARMENCITA',
            codigoBD: '01732500SKY6C21',
            tablas: ['slave3', 'auxiliar1', 'auxiliar2', 'slave1'],
        },
        CRETA: {
            nombre: 'CRETA',
            codigoBD: '01864308SKYFA81',
            tablas: ['slave3', 'auxiliar1', 'auxiliar2', 'slave1'],
        },
        RODAS: {
            nombre: 'RODAS',
            codigoBD: '01864316SKY1AA9',
            tablas: ['slave3', 'auxiliar1', 'auxiliar2', 'slave1'],
        },
        CUZCO4: {
            nombre: 'CUZCO4',
            codigoBD: '01739954SKY75B7',
            tablas: ['reportesmelecto', 'auxiliar1', 'slave1'],
        },
        ANCASH2: {
            nombre: 'ANCASH2',
            codigoBD: '01863868SKY11E9',
            tablas: ['reportesmelecto', 'slave1', 'auxiliar1'],
        },
        CARIBE: {
            nombre: 'CARIBE',
            codigoBD: '01343906SKYFE67',
            tablas: ['reportesmelecto'],
        },
    };
    sistemaTablasMap = {
        reportesmelecto: ['reportesmelecto'],
        slave1: ['slave1'],
        slave3: ['slave3'],
        slave5: ['slave5'],
        auxiliar1: ['auxiliar1'],
        auxiliar2: ['auxiliar2'],
        auxiliar3: ['auxiliar3'],
    };
    constructor(exalmarDatabaseService) {
        this.exalmarDatabaseService = exalmarDatabaseService;
    }
    generateCacheKey(dto) {
        return `exalmar_${dto.sistema || 'all'}_${dto.embarcacion || 'all'}_${dto.desde || ''}_${dto.hasta || ''}_${dto.limit || ''}`;
    }
    getCachedData(key) {
        const cached = this.cache.get(key);
        if (cached && Date.now() - cached.timestamp < this.CACHE_TTL) {
            this.logger.debug(`Cache hit para: ${key}`);
            return cached.data;
        }
        if (cached) {
            this.cache.delete(key);
        }
        return null;
    }
    setCachedData(key, data) {
        this.cache.set(key, { data, timestamp: Date.now() });
        this.logger.debug(`Cache set para: ${key}`);
    }
    async executeWithLimitedConcurrency(tasks, concurrencyLimit = 5) {
        const results = [];
        for (let i = 0; i < tasks.length; i += concurrencyLimit) {
            const batch = tasks.slice(i, i + concurrencyLimit);
            const batchResults = await Promise.allSettled(batch.map((task) => task()));
            batchResults.forEach((result, index) => {
                if (result.status === 'fulfilled') {
                    results.push(result.value);
                }
                else {
                    this.logger.warn(`Error en tarea ${i + index}:`, result.reason);
                }
            });
        }
        return results;
    }
    async consultarRegistros(dto) {
        try {
            this.logger.log(`Iniciando consulta EXALMAR: ${JSON.stringify(dto)}`);
            const cacheKey = this.generateCacheKey(dto);
            const cachedResult = this.getCachedData(cacheKey);
            if (cachedResult) {
                return cachedResult;
            }
            let resultado;
            if (dto.embarcacion) {
                resultado = await this.consultarEmbarcacionEspecifica(dto);
            }
            else if (dto.sistema) {
                resultado = await this.consultarSistemaEspecifico(dto);
            }
            else {
                resultado = await this.consultarTodasLasEmbarcaciones(dto);
            }
            await this.exalmarDatabaseService.forceCloseAllConnections();
            this.setCachedData(cacheKey, resultado);
            return resultado;
        }
        catch (error) {
            this.logger.error(`Error en consulta EXALMAR: ${error.message}`, error.stack);
            await this.exalmarDatabaseService.forceCloseAllConnections();
            throw new common_1.InternalServerErrorException('Error al consultar registros EXALMAR');
        }
    }
    async consultarEmbarcacionEspecifica(dto) {
        if (!dto.embarcacion) {
            throw new common_1.BadRequestException('Embarcación es requerida para consulta específica');
        }
        const embarcacionConfig = this.embarcacionesConfig[dto.embarcacion.toUpperCase()];
        if (!embarcacionConfig) {
            throw new common_1.BadRequestException(`Embarcación ${dto.embarcacion} no encontrada`);
        }
        const tablasAConsultar = dto.sistema
            ? embarcacionConfig.tablas.filter((tabla) => this.sistemaTablasMap[dto.sistema]?.includes(tabla))
            : embarcacionConfig.tablas;
        if (tablasAConsultar.length === 0) {
            this.logger.warn(`No se encontraron tablas para ${dto.embarcacion} con sistema ${dto.sistema}`);
            return { [dto.embarcacion]: [] };
        }
        const tareas = tablasAConsultar.map((tabla) => () => this.traerRegistrosEmbarcacion(embarcacionConfig, tabla, dto)
            .then((registros) => ({
            embarcacion: dto.embarcacion,
            tabla,
            registros,
        }))
            .catch((error) => {
            this.logger.warn(`Error consultando tabla ${tabla} de ${dto.embarcacion}: ${error.message}`);
            return { embarcacion: dto.embarcacion, tabla, registros: [] };
        }));
        this.logger.log(`Ejecutando ${tareas.length} consultas para embarcación ${dto.embarcacion}`);
        const resultados = await this.executeWithLimitedConcurrency(tareas, 5);
        const isDateRangeQuery = !!(dto.desde || dto.hasta);
        if (dto.limit && dto.limit > 1) {
            return this.agruparPorEmbarcacionConLimitePorTabla(resultados, dto.limit);
        }
        return this.agruparPorEmbarcacion(resultados, dto.limit, isDateRangeQuery);
    }
    async consultarSistemaEspecifico(dto) {
        if (!dto.sistema) {
            throw new common_1.BadRequestException('Sistema es requerido para consulta específica');
        }
        const tablasDelSistema = this.sistemaTablasMap[dto.sistema];
        if (!tablasDelSistema) {
            throw new common_1.BadRequestException(`Sistema ${dto.sistema} no encontrado`);
        }
        const tareas = [];
        for (const [nombreEmbarcacion, config] of Object.entries(this.embarcacionesConfig)) {
            const tablasComunes = config.tablas.filter((tabla) => tablasDelSistema.includes(tabla));
            for (const tabla of tablasComunes) {
                tareas.push(() => this.traerRegistrosEmbarcacion(config, tabla, dto)
                    .then((registros) => ({
                    embarcacion: nombreEmbarcacion,
                    tabla,
                    registros,
                }))
                    .catch((error) => {
                    this.logger.warn(`Error consultando ${nombreEmbarcacion}/${tabla}: ${error.message}`);
                    return { embarcacion: nombreEmbarcacion, tabla, registros: [] };
                }));
            }
        }
        this.logger.log(`Ejecutando ${tareas.length} consultas para sistema ${dto.sistema}`);
        const resultados = await this.executeWithLimitedConcurrency(tareas, 5);
        const isDateRangeQuery = !!(dto.desde || dto.hasta);
        if (dto.limit && dto.limit > 1) {
            return this.agruparPorEmbarcacionConLimitePorTabla(resultados, dto.limit);
        }
        return this.agruparPorEmbarcacion(resultados, dto.limit, isDateRangeQuery);
    }
    async consultarTodasLasEmbarcaciones(dto) {
        const tareas = [];
        for (const [nombreEmbarcacion, config] of Object.entries(this.embarcacionesConfig)) {
            for (const tabla of config.tablas) {
                tareas.push(() => this.traerRegistrosEmbarcacion(config, tabla, dto)
                    .then((registros) => ({
                    embarcacion: nombreEmbarcacion,
                    tabla,
                    registros,
                }))
                    .catch((error) => {
                    this.logger.warn(`Error consultando ${nombreEmbarcacion}/${tabla}: ${error.message}`);
                    return { embarcacion: nombreEmbarcacion, tabla, registros: [] };
                }));
            }
        }
        this.logger.log(`Ejecutando ${tareas.length} consultas para todas las embarcaciones`);
        const resultados = await this.executeWithLimitedConcurrency(tareas, 5);
        const isDateRangeQuery = !!(dto.desde && dto.hasta);
        if (dto.limit && dto.limit > 1) {
            return this.agruparPorEmbarcacionConLimitePorTabla(resultados, dto.limit);
        }
        return this.agruparPorEmbarcacion(resultados, dto.limit, isDateRangeQuery);
    }
    async traerRegistrosEmbarcacion(embarcacionConfig, tabla, dto) {
        try {
            const dbKey = embarcacionConfig.nombre;
            let query = `SELECT * FROM ${tabla}`;
            const whereConditions = [];
            const replacements = {};
            if (dto.desde) {
                whereConditions.push('fecha >= :desde');
                replacements.desde = dto.desde;
            }
            if (dto.hasta) {
                whereConditions.push('fecha <= :hasta');
                replacements.hasta = dto.hasta;
            }
            if (whereConditions.length > 0) {
                query += ` WHERE ${whereConditions.join(' AND ')}`;
            }
            query += ' ORDER BY fecha DESC';
            if (dto.limit) {
                query += ` LIMIT :limit`;
                replacements.limit = dto.limit;
            }
            this.logger.debug(`Consultando ${dbKey}.${tabla}: ${query}`);
            const connection = await this.exalmarDatabaseService.getConnection('EXALMAR', dbKey);
            const registros = await connection.query(query, {
                type: sequelize_1.QueryTypes.SELECT,
                replacements,
            });
            return registros.map((registro) => this.procesarRegistro(registro, tabla, embarcacionConfig));
        }
        catch (error) {
            this.logger.error(`Error consultando ${embarcacionConfig.nombre}/${tabla}: ${error.message}`);
            throw error;
        }
    }
    agruparPorEmbarcacion(resultados, limit, isDateRangeQuery = false) {
        const agrupado = {};
        for (const resultado of resultados) {
            if (!agrupado[resultado.embarcacion]) {
                agrupado[resultado.embarcacion] = [];
            }
            let registrosAIncluir;
            if (isDateRangeQuery) {
                registrosAIncluir = resultado.registros;
            }
            else if (!limit) {
                registrosAIncluir = resultado.registros.slice(0, 1);
            }
            else {
                registrosAIncluir = resultado.registros.slice(0, limit);
            }
            agrupado[resultado.embarcacion].push(...registrosAIncluir);
        }
        for (const embarcacion in agrupado) {
            agrupado[embarcacion] = this.ordenarRegistrosPorReporteYFecha(agrupado[embarcacion]);
        }
        const embarcacionesOrdenadas = Object.keys(agrupado).sort();
        const resultadoOrdenado = {};
        for (const embarcacion of embarcacionesOrdenadas) {
            resultadoOrdenado[embarcacion] = agrupado[embarcacion];
        }
        this.logger.log(`Resultados agrupados en ${Object.keys(resultadoOrdenado).length} embarcaciones (ordenadas)`);
        return resultadoOrdenado;
    }
    agruparPorEmbarcacionConLimitePorTabla(resultados, limit) {
        const agrupado = {};
        const agrupadoPorTabla = {};
        for (const resultado of resultados) {
            if (!agrupadoPorTabla[resultado.embarcacion]) {
                agrupadoPorTabla[resultado.embarcacion] = {};
            }
            if (!agrupadoPorTabla[resultado.embarcacion][resultado.tabla]) {
                agrupadoPorTabla[resultado.embarcacion][resultado.tabla] = [];
            }
            agrupadoPorTabla[resultado.embarcacion][resultado.tabla].push(...resultado.registros);
        }
        for (const embarcacion in agrupadoPorTabla) {
            agrupado[embarcacion] = [];
            for (const tabla in agrupadoPorTabla[embarcacion]) {
                const registrosTabla = agrupadoPorTabla[embarcacion][tabla].sort((a, b) => {
                    const fechaA = new Date(a.fecha).getTime();
                    const fechaB = new Date(b.fecha).getTime();
                    return fechaB - fechaA;
                });
                const registrosLimitados = registrosTabla.slice(0, limit);
                agrupado[embarcacion].push(...registrosLimitados);
            }
            agrupado[embarcacion] = this.ordenarRegistrosPorReporteYFecha(agrupado[embarcacion]);
        }
        const embarcacionesOrdenadas = Object.keys(agrupado).sort();
        const resultadoOrdenado = {};
        for (const embarcacion of embarcacionesOrdenadas) {
            resultadoOrdenado[embarcacion] = agrupado[embarcacion];
        }
        this.logger.log(`Resultados agrupados con límite ${limit} por tabla en ${Object.keys(resultadoOrdenado).length} embarcaciones (ordenadas)`);
        return resultadoOrdenado;
    }
    ordenarRegistrosPorReporteYFecha(registros) {
        const prioridadReportes = {
            reportesmelecto: 1,
            slave1: 2,
            slave3: 3,
            slave4: 4,
            slave5: 5,
            slave8: 6,
            slave9: 7,
            auxiliar1: 8,
            auxiliar2: 9,
            auxiliar3: 10,
            restfull: 11,
        };
        return registros.sort((a, b) => {
            const reporteA = a.tabla || a.Tabla;
            const reporteB = b.tabla || b.Tabla;
            const prioridadA = prioridadReportes[reporteA] || 999;
            const prioridadB = prioridadReportes[reporteB] || 999;
            if (prioridadA !== prioridadB) {
                return prioridadA - prioridadB;
            }
            const fechaA = new Date(a.fecha).getTime();
            const fechaB = new Date(b.fecha).getTime();
            return fechaB - fechaA;
        });
    }
    async healthCheck() {
        const connections = {};
        let healthyCount = 0;
        for (const [nombre, config] of Object.entries(this.embarcacionesConfig)) {
            try {
                const dbKey = config.nombre;
                const connection = await this.exalmarDatabaseService.getConnection('EXALMAR', dbKey);
                await connection.query('SELECT 1', { type: sequelize_1.QueryTypes.SELECT });
                connections[nombre] = true;
                healthyCount++;
            }
            catch (error) {
                this.logger.warn(`Health check falló para ${nombre}: ${error.message}`);
                connections[nombre] = false;
            }
        }
        const totalDatabases = Object.keys(this.embarcacionesConfig).length;
        const status = healthyCount === totalDatabases
            ? 'healthy'
            : healthyCount > 0
                ? 'degraded'
                : 'unhealthy';
        return {
            status,
            connections,
            timestamp: new Date().toISOString(),
            totalDatabases,
            healthyDatabases: healthyCount,
        };
    }
    async getEmbarcacionesDisponibles() {
        return Object.keys(this.embarcacionesConfig);
    }
    async getSistemasDisponibles() {
        return Object.keys(this.sistemaTablasMap);
    }
    async getBasesDatosDisponibles() {
        return this.exalmarDatabaseService.getAvailableDatabases();
    }
    procesarRegistro(row, tabla, embarcacionConfig) {
        switch (tabla) {
            case 'reportesmelecto':
                return this.procesarReportesmelecto(row, tabla, embarcacionConfig);
            case 'slave1':
                return this.procesarSlave1(row, tabla, embarcacionConfig);
            case 'slave3':
                return this.procesarSlave3(row, tabla, embarcacionConfig);
            case 'auxiliar1':
            case 'auxiliar2':
            case 'auxiliar3':
                return this.procesarAuxiliar(row, tabla, embarcacionConfig);
            case 'slave5':
                return this.procesarSlave5(row, tabla, embarcacionConfig);
            default:
                return this.procesarRegistroGenerico(row, tabla, embarcacionConfig);
        }
    }
    procesarRegistroGenerico(row, tabla, embarcacionConfig) {
        return {
            idmensaje: embarcacionConfig.codigoBD,
            Reporte: tabla,
            Tabla: embarcacionConfig.nombre,
            fecha: row.fecha,
            latitud: row.latitude,
            longitud: row.longitude,
            speed: row.speed,
            header: row.heading ?? row.header,
            ...row,
            id: embarcacionConfig.codigoBD,
            embarcacion: embarcacionConfig.nombre,
            tabla: tabla,
        };
    }
    procesarReportesmelecto(row, tabla, embarcacionConfig) {
        return {
            id: embarcacionConfig.codigoBD,
            fecha: row.fecha,
            latitud: row.latitude,
            longitud: row.longitude,
            velocidad: row.speed,
            rumbo: row.header,
            load_rpm: row.parameter01,
            rpm: row.parameter02,
            horometro: row.parameter04,
            consumo_total: row.parameter06,
            fuel_rate: row.parameter12,
            tabla: 'restfull',
        };
    }
    procesarSlave1(row, tabla, embarcacionConfig) {
        return {
            id: embarcacionConfig.codigoBD,
            fecha: row.fecha,
            latitud: row.latitude,
            longitud: row.longitude,
            speed: row.speed,
            header: row.heading ?? row.header,
            PP_BR_D: row.PR_BR,
            PP_BR_S: row.PR_ER,
            PR_BR_D: row.PR_CT,
            PR_BR_S: row.PP_CT,
            PP_CT_D: row.PP_BR,
            PP_CT_S: row.PP_ER,
            PR_CT_D: row.TEMP_1,
            PR_CT_S: row.TEMP_2,
            PR_ER_D: row.TEMP_3,
            PR_ER_S: row.TEMP_4,
            PP_ER_D: row.TEMP_5,
            PP_ER_S: row.TEMP_6,
            tabla: tabla,
        };
    }
    procesarSlave3(row, tabla, embarcacionConfig) {
        return {
            id: embarcacionConfig.codigoBD,
            fecha: row.fecha,
            latitud: row.latitude,
            longitud: row.longitude,
            velocidad: row.speed,
            rumbo: row.heading ?? row.header,
            temp_ingreso: row.TEMP_IN,
            temp_retorno: row.TEMP_RET,
            rpm: row.RPM,
            flujo_ingreso: row.FR_IN,
            flujo_retorno: row.FR_RET,
            consumo_h: row.GAL_H,
            consumo_total: row.TOTAL,
            horometro: row.HOROM,
            tabla: tabla,
        };
    }
    procesarAuxiliar(row, tabla, embarcacionConfig) {
        if (embarcacionConfig.codigoBD == '01863868SKY11E9' ||
            embarcacionConfig.codigoBD == '01739954SKY75B7') {
            if (tabla == 'auxiliar1') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.header,
                    load_rpm: row.parameter01,
                    rpm: row.parameter02,
                    horometro: row.parameter04,
                    consumo_total: row.parameter06,
                    fuel_rate: row.parameter12,
                    tabla: 'slave9',
                };
            }
        }
        else if (embarcacionConfig.codigoBD == '01864307SKY767C' ||
            embarcacionConfig.codigoBD == '01864316SKY1AA9' ||
            embarcacionConfig.codigoBD == '01732500SKY6C21' ||
            embarcacionConfig.codigoBD == '01864308SKYFA81') {
            if (tabla == 'auxiliar1') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.header,
                    load_rpm: row.parameter01,
                    rpm: row.parameter02,
                    horometro: row.parameter04,
                    consumo_total: row.parameter06,
                    fuel_rate: row.parameter12,
                    tabla: 'slave8',
                };
            }
            else if (tabla == 'auxiliar2') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.header,
                    load_rpm: row.parameter01,
                    rpm: row.parameter02,
                    horometro: row.parameter04,
                    consumo_total: row.parameter06,
                    fuel_rate: row.parameter12,
                    tabla: 'slave9',
                };
            }
            else if (tabla == 'auxiliar3') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.heading,
                    temp_ingreso: row.TEMP_IN,
                    temp_retorno: row.TEMP_RET,
                    rpm: row.RPM,
                    flujo_ingreso: row.FR_IN,
                    flujo_retorno: row.FR_RET,
                    consumo_h: row.GAL_H,
                    consumo_total: row.TOTAL,
                    tabla: 'slave3',
                };
            }
        }
        else if (embarcacionConfig.codigoBD == '01864306SKYF277') {
            if (tabla == 'auxiliar1') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.heading,
                    temp_ingreso: row.TEMP_IN,
                    temp_retorno: row.TEMP_RET,
                    rpm: row.RPM,
                    flujo_ingreso: row.FR_IN,
                    flujo_retorno: row.FR_RET,
                    consumo_h: row.GAL_H,
                    consumo_total: row.TOTAL,
                    tabla: 'slave4',
                };
            }
            else if (tabla == 'auxiliar2') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.heading,
                    temp_ingreso: row.TEMP_IN,
                    temp_retorno: row.TEMP_RET,
                    rpm: row.RPM,
                    flujo_ingreso: row.FR_IN,
                    flujo_retorno: row.FR_RET,
                    consumo_h: row.GAL_H,
                    consumo_total: row.TOTAL,
                    tabla: 'slave8',
                };
            }
            else if (tabla == 'auxiliar3') {
                return {
                    id: embarcacionConfig.codigoBD,
                    fecha: row.fecha,
                    latitud: row.latitude,
                    longitud: row.longitude,
                    velocidad: row.speed,
                    rumbo: row.heading,
                    temp_ingreso: row.TEMP_IN,
                    temp_retorno: row.TEMP_RET,
                    rpm: row.RPM,
                    flujo_ingreso: row.FR_IN,
                    flujo_retorno: row.FR_RET,
                    consumo_h: row.GAL_H,
                    consumo_total: row.TOTAL,
                    tabla: 'slave9',
                };
            }
        }
    }
    procesarSlave5(row, tabla, embarcacionConfig) {
        return {
            id: embarcacionConfig.codigoBD,
            fecha: row.fecha,
            latitud: row.latitude,
            longitud: row.longitude,
            velocidad: row.speed,
            rumbo: row.heading ?? row.header,
            mp_diario: row.TK_1,
            taller_er: row.TK_2,
            sm_estribor: row.TK_3,
            lazareto_er: row.TK_4,
            grupos: row.TK_5,
            taller_babor: row.TK_6,
            sm_babor: row.TK_7,
            tabla: tabla,
        };
    }
};
exports.ExalmarService = ExalmarService;
exports.ExalmarService = ExalmarService = ExalmarService_1 = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [exalmar_database_service_1.ExalmarDatabaseService])
], ExalmarService);
//# sourceMappingURL=exalmar.service.js.map