"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var ExalmarController_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExalmarController = void 0;
const common_1 = require("@nestjs/common");
const swagger_1 = require("@nestjs/swagger");
const exalmar_service_1 = require("./exalmar.service");
const consulta_exalmar_dto_1 = require("./dto/consulta-exalmar.dto");
let ExalmarController = ExalmarController_1 = class ExalmarController {
    exalmarService;
    logger = new common_1.Logger(ExalmarController_1.name);
    constructor(exalmarService) {
        this.exalmarService = exalmarService;
    }
    async healthCheck() {
        this.logger.log('Ejecutando health check para EXALMAR');
        return await this.exalmarService.healthCheck();
    }
    async getEmbarcacionesDisponibles() {
        this.logger.log('Obteniendo embarcaciones disponibles');
        return await this.exalmarService.getEmbarcacionesDisponibles();
    }
    async getSistemasDisponibles() {
        this.logger.log('Obteniendo sistemas disponibles para EXALMAR');
        return await this.exalmarService.getSistemasDisponibles();
    }
    async getBasesDatosDisponibles() {
        this.logger.log('Obteniendo bases de datos disponibles para EXALMAR');
        return await this.exalmarService.getBasesDatosDisponibles();
    }
    async consultarRegistros(dto) {
        this.logger.log(`Consultando registros EXALMAR con parámetros: ${JSON.stringify(dto)}`);
        try {
            const resultado = await this.exalmarService.consultarRegistros(dto);
            const totalRegistros = Object.values(resultado).reduce((total, registros) => total + registros.length, 0);
            this.logger.log(`Consulta EXALMAR completada: ${totalRegistros} registros en ${Object.keys(resultado).length} embarcaciones`);
            return resultado;
        }
        catch (error) {
            this.logger.error('Error en consulta EXALMAR:', error);
            throw error;
        }
    }
    async ruta1_ultimoRegistroGeneral() {
        this.logger.log('Ruta 1: Obteniendo último registro general EXALMAR');
        return await this.exalmarService.consultarRegistros({ limit: 1 });
    }
    async ruta3_ultimoRegistroSistema(sistema) {
        this.logger.log(`Ruta 3: Obteniendo último registro del sistema ${sistema} en EXALMAR`);
        return await this.exalmarService.consultarRegistros({
            sistema: sistema.toLowerCase(),
            limit: 1,
        });
    }
    async ruta4_registrosSistema(sistema, limit) {
        const limitNum = parseInt(limit);
        if (isNaN(limitNum) || limitNum <= 0 || limitNum > 100) {
            throw new common_1.HttpException('El límite debe ser un número entre 1 y 100', common_1.HttpStatus.BAD_REQUEST);
        }
        this.logger.log(`Ruta 4: Obteniendo últimos ${limit} registros del sistema ${sistema} en EXALMAR`);
        return await this.exalmarService.consultarRegistros({
            sistema: sistema.toLowerCase(),
            limit: limitNum,
        });
    }
    async ruta5_ultimoRegistroEmbarcacion(embarcacion) {
        const embarcacionUpper = embarcacion.toUpperCase();
        const embarcacionesValidas = await this.exalmarService.getEmbarcacionesDisponibles();
        if (!embarcacionesValidas.includes(embarcacionUpper)) {
            throw new common_1.HttpException(`Embarcación '${embarcacion}' no está configurada`, common_1.HttpStatus.BAD_REQUEST);
        }
        this.logger.log(`Ruta 5: Obteniendo último registro de la embarcación ${embarcacionUpper}`);
        return await this.exalmarService.consultarRegistros({
            embarcacion: embarcacionUpper,
            limit: 1,
        });
    }
    async ruta6_registrosEmbarcacion(embarcacion, limit) {
        const limitNum = parseInt(limit);
        if (isNaN(limitNum) || limitNum <= 0 || limitNum > 100) {
            throw new common_1.HttpException('El límite debe ser un número entre 1 y 100', common_1.HttpStatus.BAD_REQUEST);
        }
        const embarcacionUpper = embarcacion.toUpperCase();
        const embarcacionesValidas = await this.exalmarService.getEmbarcacionesDisponibles();
        if (!embarcacionesValidas.includes(embarcacionUpper)) {
            throw new common_1.HttpException(`Embarcación '${embarcacion}' no está configurada`, common_1.HttpStatus.BAD_REQUEST);
        }
        this.logger.log(`Ruta 6: Obteniendo últimos ${limit} registros de la embarcación ${embarcacionUpper}`);
        return await this.exalmarService.consultarRegistros({
            embarcacion: embarcacionUpper,
            limit: limitNum,
        });
    }
    async ruta7_consultaCompletaEmbarcacion(embarcacion, fechaInicio, fechaFin, sistema) {
        if (!this.isValidDate(fechaInicio) || !this.isValidDate(fechaFin)) {
            throw new common_1.HttpException('Fechas inválidas. Deben tener formato YYYY-MM-DD', common_1.HttpStatus.BAD_REQUEST);
        }
        this.validateDateRange(fechaInicio, fechaFin, 10);
        const embarcacionUpper = embarcacion.toUpperCase();
        const embarcacionesValidas = await this.exalmarService.getEmbarcacionesDisponibles();
        if (!embarcacionesValidas.includes(embarcacionUpper)) {
            throw new common_1.HttpException(`Embarcación '${embarcacion}' no está configurada`, common_1.HttpStatus.BAD_REQUEST);
        }
        this.logger.log(`Ruta 7: Consultando ${sistema} en ${embarcacionUpper} desde ${fechaInicio} hasta ${fechaFin}`);
        return await this.exalmarService.consultarRegistros({
            embarcacion: embarcacionUpper,
            desde: fechaInicio,
            hasta: `${fechaFin} 23:59:59`,
            sistema: sistema.toLowerCase(),
        });
    }
    async ruta8_consultaPorFechas(fechaInicio, fechaFin) {
        if (!this.isValidDate(fechaInicio) || !this.isValidDate(fechaFin)) {
            throw new common_1.HttpException('Fechas inválidas. Deben tener formato YYYY-MM-DD', common_1.HttpStatus.BAD_REQUEST);
        }
        this.validateDateRange(fechaInicio, fechaFin, 5);
        this.logger.log(`Ruta 8: Consultando desde ${fechaInicio} hasta ${fechaFin} en todas las embarcaciones EXALMAR`);
        return await this.exalmarService.consultarRegistros({
            desde: fechaInicio,
            hasta: `${fechaFin} 23:59:59`,
        });
    }
    async ruta9_consultaPorFechasSistema(fechaInicio, fechaFin, sistema) {
        if (!this.isValidDate(fechaInicio) || !this.isValidDate(fechaFin)) {
            throw new common_1.HttpException('Fechas inválidas. Deben tener formato YYYY-MM-DD', common_1.HttpStatus.BAD_REQUEST);
        }
        this.validateDateRange(fechaInicio, fechaFin, 10);
        this.logger.log(`Ruta 9: Consultando ${sistema} desde ${fechaInicio} hasta ${fechaFin} en todas las embarcaciones EXALMAR`);
        return await this.exalmarService.consultarRegistros({
            desde: fechaInicio,
            hasta: `${fechaFin} 23:59:59`,
            sistema: sistema.toLowerCase(),
        });
    }
    async ruta2_registrosGenerales(limit) {
        const limitNum = parseInt(limit);
        if (isNaN(limitNum) || limitNum <= 0 || limitNum > 100) {
            throw new common_1.HttpException('El límite debe ser un número entre 1 y 100', common_1.HttpStatus.BAD_REQUEST);
        }
        this.logger.log(`Ruta 2: Obteniendo últimos ${limit} registros generales EXALMAR`);
        return await this.exalmarService.consultarRegistros({ limit: limitNum });
    }
    isValidDate(dateString) {
        const regex = /^\d{4}-\d{2}-\d{2}$/;
        if (!regex.test(dateString))
            return false;
        const date = new Date(dateString);
        return !isNaN(date.getTime());
    }
    validateDateRange(fechaInicio, fechaFin, maxDays) {
        const startDate = new Date(fechaInicio);
        const endDate = new Date(fechaFin);
        if (startDate > endDate) {
            throw new common_1.HttpException('La fecha de inicio no puede ser posterior a la fecha de fin', common_1.HttpStatus.BAD_REQUEST);
        }
        const diffTime = Math.abs(endDate.getTime() - startDate.getTime());
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        if (diffDays > maxDays) {
            throw new common_1.HttpException(`El rango de fechas no puede ser mayor a ${maxDays} días`, common_1.HttpStatus.BAD_REQUEST);
        }
    }
};
exports.ExalmarController = ExalmarController;
__decorate([
    (0, common_1.Get)('health'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiExcludeEndpoint)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "healthCheck", null);
__decorate([
    (0, common_1.Get)('embarcaciones'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiExcludeEndpoint)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "getEmbarcacionesDisponibles", null);
__decorate([
    (0, common_1.Get)('sistemas'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiExcludeEndpoint)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "getSistemasDisponibles", null);
__decorate([
    (0, common_1.Get)('bases-datos'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiExcludeEndpoint)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "getBasesDatosDisponibles", null);
__decorate([
    (0, common_1.Get)('consultar'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiExcludeEndpoint)(),
    __param(0, (0, common_1.Query)()),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [consulta_exalmar_dto_1.ConsultaGeneralExalmarDto]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "consultarRegistros", null);
__decorate([
    (0, common_1.Get)(),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '1. Último registro general',
        description: 'Obtiene el último registro de todas las embarcaciones y sistemas',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Último registro obtenido exitosamente',
        schema: {
            type: 'object',
            additionalProperties: {
                type: 'array',
                items: { type: 'object' },
            },
        },
    }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta1_ultimoRegistroGeneral", null);
__decorate([
    (0, common_1.Get)('sistema/:sistema'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '3. Último registro de sistema específico',
        description: 'Obtiene el último registro de un sistema específico en todas las embarcaciones que lo tengan',
    }),
    (0, swagger_1.ApiParam)({
        name: 'sistema',
        description: 'Sistema específico a consultar',
        example: 'slave1',
        enum: [
            'reportesmelecto',
            'slave1',
            'slave3',
            'slave5',
            'auxiliar1',
            'auxiliar2',
            'auxiliar3',
        ],
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Último registro del sistema obtenido exitosamente',
    }),
    __param(0, (0, common_1.Param)('sistema')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta3_ultimoRegistroSistema", null);
__decorate([
    (0, common_1.Get)('sistema/:sistema/:limit'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '4. N registros de sistema específico',
        description: 'Obtiene los últimos N registros de un sistema específico en todas las embarcaciones',
    }),
    (0, swagger_1.ApiParam)({
        name: 'sistema',
        description: 'Sistema específico a consultar',
        example: 'slave1',
    }),
    (0, swagger_1.ApiParam)({
        name: 'limit',
        description: 'Número de registros a obtener (1-100)',
        example: '10',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Registros del sistema obtenidos exitosamente',
    }),
    __param(0, (0, common_1.Param)('sistema')),
    __param(1, (0, common_1.Param)('limit')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String, String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta4_registrosSistema", null);
__decorate([
    (0, common_1.Get)('embarcacion/:embarcacion'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '5. Último registro de embarcación específica',
        description: 'Obtiene el último registro de una embarcación específica (ej: DONALFREDO, MARIANGELLA)',
    }),
    (0, swagger_1.ApiParam)({
        name: 'embarcacion',
        description: 'Embarcación específica',
        example: 'DONALFREDO',
        enum: [
            'DONALFREDO',
            'MARIANGELLA',
            'CARMENCITA',
            'CRETA',
            'RODAS',
            'CUZCO4',
            'ANCASH2',
            'CARIBE',
        ],
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Último registro de la embarcación obtenido exitosamente',
    }),
    __param(0, (0, common_1.Param)('embarcacion')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta5_ultimoRegistroEmbarcacion", null);
__decorate([
    (0, common_1.Get)('embarcacion/:embarcacion/:limit'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '6. N registros de embarcación específica',
        description: 'Obtiene los últimos N registros de una embarcación específica',
    }),
    (0, swagger_1.ApiParam)({
        name: 'embarcacion',
        description: 'Embarcación específica',
        example: 'DONALFREDO',
    }),
    (0, swagger_1.ApiParam)({
        name: 'limit',
        description: 'Número de registros a obtener (1-100)',
        example: '10',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Registros de la embarcación obtenidos exitosamente',
    }),
    __param(0, (0, common_1.Param)('embarcacion')),
    __param(1, (0, common_1.Param)('limit')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String, String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta6_registrosEmbarcacion", null);
__decorate([
    (0, common_1.Get)('embarcacion/:embarcacion/:fechaInicio/:fechaFin/:sistema'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '7. Consulta completa en embarcación específica',
        description: 'Consulta registros de un sistema específico en un rango de fechas en una embarcación específica',
    }),
    (0, swagger_1.ApiParam)({
        name: 'embarcacion',
        description: 'Embarcación específica',
        example: 'DONALFREDO',
    }),
    (0, swagger_1.ApiParam)({
        name: 'fechaInicio',
        description: 'Fecha de inicio (YYYY-MM-DD)',
        example: '2025-01-01',
    }),
    (0, swagger_1.ApiParam)({
        name: 'fechaFin',
        description: 'Fecha de fin (YYYY-MM-DD)',
        example: '2025-01-31',
    }),
    (0, swagger_1.ApiParam)({
        name: 'sistema',
        description: 'Sistema específico',
        example: 'slave1',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Registros filtrados obtenidos exitosamente',
    }),
    __param(0, (0, common_1.Param)('embarcacion')),
    __param(1, (0, common_1.Param)('fechaInicio')),
    __param(2, (0, common_1.Param)('fechaFin')),
    __param(3, (0, common_1.Param)('sistema')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String, String, String, String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta7_consultaCompletaEmbarcacion", null);
__decorate([
    (0, common_1.Get)('fechas/:fechaInicio/:fechaFin'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '8. Consulta por fechas en todas las embarcaciones',
        description: 'Consulta registros en un rango de fechas en todas las embarcaciones disponibles',
    }),
    (0, swagger_1.ApiParam)({
        name: 'fechaInicio',
        description: 'Fecha de inicio (YYYY-MM-DD)',
        example: '2025-01-01',
    }),
    (0, swagger_1.ApiParam)({
        name: 'fechaFin',
        description: 'Fecha de fin (YYYY-MM-DD)',
        example: '2025-01-31',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Registros en rango de fechas obtenidos exitosamente',
    }),
    __param(0, (0, common_1.Param)('fechaInicio')),
    __param(1, (0, common_1.Param)('fechaFin')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String, String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta8_consultaPorFechas", null);
__decorate([
    (0, common_1.Get)('fechas/:fechaInicio/:fechaFin/:sistema'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '9. Consulta por fechas y sistema en todas las embarcaciones',
        description: 'Consulta registros de un sistema específico en un rango de fechas en todas las embarcaciones',
    }),
    (0, swagger_1.ApiParam)({
        name: 'fechaInicio',
        description: 'Fecha de inicio (YYYY-MM-DD)',
        example: '2025-01-01',
    }),
    (0, swagger_1.ApiParam)({
        name: 'fechaFin',
        description: 'Fecha de fin (YYYY-MM-DD)',
        example: '2025-01-31',
    }),
    (0, swagger_1.ApiParam)({
        name: 'sistema',
        description: 'Sistema específico',
        example: 'slave1',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Registros filtrados por fechas y sistema obtenidos exitosamente',
    }),
    __param(0, (0, common_1.Param)('fechaInicio')),
    __param(1, (0, common_1.Param)('fechaFin')),
    __param(2, (0, common_1.Param)('sistema')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String, String, String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta9_consultaPorFechasSistema", null);
__decorate([
    (0, common_1.Get)(':limit'),
    (0, common_1.HttpCode)(common_1.HttpStatus.OK),
    (0, swagger_1.ApiOperation)({
        summary: '2. N registros generales',
        description: 'Obtiene los últimos N registros de todas las embarcaciones y sistemas',
    }),
    (0, swagger_1.ApiParam)({
        name: 'limit',
        description: 'Número de registros a obtener (1-100)',
        example: '10',
    }),
    (0, swagger_1.ApiResponse)({
        status: 200,
        description: 'Registros generales obtenidos exitosamente',
    }),
    __param(0, (0, common_1.Param)('limit')),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], ExalmarController.prototype, "ruta2_registrosGenerales", null);
exports.ExalmarController = ExalmarController = ExalmarController_1 = __decorate([
    (0, swagger_1.ApiTags)('EXALMAR - Consultas de Embarcaciones'),
    (0, common_1.Controller)('exalmar'),
    __metadata("design:paramtypes", [exalmar_service_1.ExalmarService])
], ExalmarController);
//# sourceMappingURL=exalmar.controller.js.map