#!/usr/bin/env bash
set -euo pipefail

# 🚀 Gestión de Microservicios - Puertos 4000, 4002, 4003
# ======================================================

# Detectar directorio base de servicios
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
DEFAULT_BASE_PARENT="$(cd "${SCRIPT_DIR}/.." && pwd)"
DEFAULT_BASE_SCRIPT="${SCRIPT_DIR}"
# Permitir override por variable de entorno SERVICES_ROOT
BASE_DIR="${SERVICES_ROOT:-${DEFAULT_BASE_PARENT}}"
# Si no existe api-gateway en BASE_DIR, asumir que los servicios están junto al script
if [[ ! -d "${BASE_DIR}/api-gateway" ]]; then
  BASE_DIR="${DEFAULT_BASE_SCRIPT}"
fi

# Servicios y puertos
declare -A SERVICES=(
  [api-gateway]=4000
  [tasa-service]=4002
  [exalmar-service]=4003
)

# Verifica PM2 e instala si no existe
ensure_pm2() {
  if ! command -v pm2 >/dev/null 2>&1; then
    echo "📦 Instalando PM2 global..."
    npm install -g pm2
  fi
}

# Evitar instalación global de NestJS CLI; usaremos npx durante el build
# ensure_nest_cli() { :; }

# Copia .env.production a .env si no existe .env
ensure_env() {
  local svc="$1"
  local dir="${BASE_DIR}/${svc}"
  if [[ -f "${dir}/.env.production" && ! -f "${dir}/.env" ]]; then
    echo "🔐 Preparando entorno para ${svc} (.env desde .env.production)"
    cp "${dir}/.env.production" "${dir}/.env"
  fi
}

# Instala dependencias, compila y arranca con PM2
install_service() {
  local svc="$1"
  local port="$2"
  local dir="${BASE_DIR}/${svc}"
  echo "🔧 Instalando ${svc} en puerto ${port}..."

  if [[ ! -d "${dir}" ]]; then
    echo "❌ Directorio no encontrado: ${dir}"
    exit 1
  fi

  pushd "${dir}" >/dev/null
  # Instalación de dependencias con fallback inteligente
  if [[ -f package-lock.json ]]; then
    if npm ci --omit=dev; then
      echo "✅ npm ci completado"
    else
      echo "ℹ️ npm ci falló, usando npm install --omit=dev"
      npm install --omit=dev
    fi
  else
    echo "ℹ️ No hay package-lock.json, usando npm install --omit=dev"
    npm install --omit=dev
  fi

  echo "🏗️ Compilando ${svc}"
  # Usar npx para evitar dependencia de CLI global
  if ! npx nest build; then
    echo "❌ Falló la compilación de ${svc} con npx nest. Intentando npm run build..."
    if ! npm run build; then
      echo "❌ Falló la compilación de ${svc}. Intentando usar salida existente..."
      if [[ -f "dist/main.js" ]]; then
        echo "ℹ️ Se encontró dist/main.js, continuando"
      else
        echo "❌ No se pudo compilar ${svc}. Revisa dependencias y CLI NestJS."
        exit 1
      fi
    fi
  fi

  ensure_env "${svc}"

  echo "🚀 Iniciando ${svc} con PM2"
  if [[ -f "dist/main.js" ]]; then
    pm2 start "node dist/main.js" --name "${svc}" --cwd "${dir}"
  else
    pm2 start "npm run start:prod" --name "${svc}" --cwd "${dir}"
  fi
  popd >/dev/null
  echo "✅ ${svc} iniciado en puerto ${port}"
}

start_service() {
  local svc="$1"
  local dir="${BASE_DIR}/${svc}"
  if ! pm2 list | grep -q "${svc}"; then
    echo "ℹ️ ${svc} no está en PM2, iniciando"
    pm2 start "npm run start:prod" --name "${svc}" --cwd "${dir}"
  else
    pm2 start "${svc}"
  fi
}

stop_service() {
  local svc="$1"
  pm2 stop "${svc}" || true
}

restart_service() {
  local svc="$1"
  pm2 restart "${svc}" || pm2 start "${svc}"
}

print_banner() {
  echo ""
  echo "🚀 Gestión de Microservicios - Puertos 4000, 4002, 4003"
  echo "=================================================="
}

usage() {
  echo "Uso: $0 {instalar|iniciar|detener|reiniciar|estado|logs|salvar}"
  echo ""
  echo "Comandos disponibles:"
  echo "  instalar  - Instalar y configurar todos los servicios"
  echo "  iniciar   - Iniciar todos los servicios"
  echo "  detener   - Detener todos los servicios"
  echo "  reiniciar - Reiniciar todos los servicios"
  echo "  estado    - Ver estado de los servicios"
  echo "  logs      - Ver logs en tiempo real"
  echo "  salvar    - Guardar configuración PM2 para arranque automático"
}

main() {
  print_banner
  ensure_pm2
  # ensure_nest_cli (omitido; usamos npx)

  local cmd="${1:-}"
  case "${cmd}" in
    instalar)
      for svc in "${!SERVICES[@]}"; do
        install_service "${svc}" "${SERVICES[${svc}]}"
      done
      echo "🎉 ¡Todos los servicios instalados!"
      ;;
    iniciar)
      for svc in "${!SERVICES[@]}"; do
        start_service "${svc}"
      done
      ;;
    detener)
      for svc in "${!SERVICES[@]}"; do
        stop_service "${svc}"
      done
      ;;
    reiniciar)
      for svc in "${!SERVICES[@]}"; do
        restart_service "${svc}"
      done
      ;;
    estado)
      pm2 status
      ;;
    logs)
      echo "Mostrando logs (Ctrl+C para salir):"
      pm2 logs
      ;;
    salvar)
      echo "Guardando configuración para inicio automático..."
      pm2 save
      pm2 startup || true
      ;;
    *)
      usage
      exit 1
      ;;
  esac
}

main "$@"
